"""Application factory and extension initialization."""
from __future__ import annotations

import logging
from logging.handlers import RotatingFileHandler
from pathlib import Path

from flask import Flask, render_template
from flask_sqlalchemy import SQLAlchemy
from flask_login import LoginManager
from flask_wtf import CSRFProtect

db = SQLAlchemy()
login_manager = LoginManager()
csrf = CSRFProtect()


def create_app(config_object: str | None = None) -> Flask:
    """Create and configure the Flask application."""
    app = Flask(
        __name__,
        instance_relative_config=True,
        template_folder="../templates",
        static_folder="../static",
    )
    Path(app.instance_path).mkdir(parents=True, exist_ok=True)

    if config_object:
        app.config.from_object(config_object)
    else:
        # Load default config then override with instance config if present.
        app.config.from_object("app.config.Config")
        app.config.from_pyfile("config.py", silent=True)

    db.init_app(app)
    login_manager.init_app(app)
    csrf.init_app(app)

    login_manager.login_view = "auth.login"
    login_manager.login_message_category = "warning"

    from .routes import auth_bp, admin_bp, user_bp  # noqa: WPS433 (local import)

    app.register_blueprint(auth_bp)
    app.register_blueprint(user_bp)
    app.register_blueprint(admin_bp)

    @app.errorhandler(404)
    def not_found(error):  # noqa: D401
        return render_template("errors/404.html"), 404

    @app.errorhandler(500)
    def internal_error(error):  # noqa: D401
        db.session.rollback()
        return render_template("errors/500.html"), 500

    if not app.debug and not app.testing:
        log_path = Path(app.instance_path) / "app.log"
        handler = RotatingFileHandler(log_path, maxBytes=10240, backupCount=5)
        handler.setLevel(logging.INFO)
        formatter = logging.Formatter(
            "%(asctime)s %(levelname)s %(name)s: %(message)s [in %(pathname)s:%(lineno)d]"
        )
        handler.setFormatter(formatter)
        app.logger.addHandler(handler)

    with app.app_context():
        db.create_all()
        _seed_data()

    return app


def _seed_data() -> None:
    """Populate the database with demo data for local testing."""
    from .models import (
        Answer,
        GradeCriteria,
        Question,
        Role,
        Test,
        TestAssignment,
        TestAttempt,
        User,
        UserAnswer,
    )

    if User.query.first():
        return

    admin = User(username="admin", role=Role.ADMIN)
    admin.set_password("Admin123!")
    user1 = User(username="ivan", role=Role.USER)
    user1.set_password("User123!")
    user2 = User(username="maria", role=Role.USER)
    user2.set_password("User123!")

    db.session.add_all([admin, user1, user2])
    db.session.commit()

    python_test = Test(
        title="Основы Python",
        description="Проверьте знания базового синтаксиса и структур данных языка Python.",
        created_by=admin.id,
        shuffle_questions=True,
        shuffle_answers=True,
    )

    q1 = Question(
        question_text="Какой тип данных возвращает функция len()?",
        question_type="single",
    )
    q1.answers = [
        Answer(answer_text="int", is_correct=True),
        Answer(answer_text="str", is_correct=False),
        Answer(answer_text="list", is_correct=False),
    ]

    q2 = Question(
        question_text="Какие конструкции применимы для цикла for?",
        question_type="multiple",
    )
    q2.answers = [
        Answer(answer_text="range()", is_correct=True),
        Answer(answer_text="enumerate()", is_correct=True),
        Answer(answer_text="append()", is_correct=False),
    ]

    python_test.questions = [q1, q2]
    python_test.grade_criteria = [
        GradeCriteria(grade=5, max_errors=0),
        GradeCriteria(grade=4, max_errors=1),
        GradeCriteria(grade=3, max_errors=2),
    ]
    python_test.assignments = [TestAssignment(assigned_to_all=True)]

    flask_test = Test(
        title="Flask для начинающих",
        description="Короткий тест по основам микрофреймворка Flask.",
        created_by=admin.id,
        shuffle_questions=False,
        shuffle_answers=True,
    )

    fq1 = Question(
        question_text="Как создать blueprint в Flask?",
        question_type="single",
    )
    fq1.answers = [
        Answer(answer_text="Blueprint('name', __name__)", is_correct=True),
        Answer(answer_text="FlaskBlueprint('name')", is_correct=False),
        Answer(answer_text="create_blueprint('name')", is_correct=False),
    ]

    fq2 = Question(
        question_text="Отметьте верные утверждения о конфигурации Flask.",
        question_type="multiple",
    )
    fq2.answers = [
        Answer(answer_text="Конфигурацию можно загрузить из объекта Python.", is_correct=True),
        Answer(answer_text="Метод app.config.from_pyfile читает файл настроек.", is_correct=True),
        Answer(answer_text="Нужно перезапускать приложение после изменения config.", is_correct=False),
    ]

    flask_test.questions = [fq1, fq2]
    flask_test.grade_criteria = [
        GradeCriteria(grade=5, max_errors=0),
        GradeCriteria(grade=4, max_errors=1),
        GradeCriteria(grade=3, max_errors=2),
    ]
    flask_test.assignments = [
        TestAssignment(user_id=user1.id, assigned_to_all=False),
        TestAssignment(user_id=user2.id, assigned_to_all=False),
    ]

    db.session.add_all([python_test, flask_test])
    db.session.commit()

    refreshed_python_test = Test.query.filter_by(title="Основы Python").first()
    if not refreshed_python_test:
        return

    attempt = TestAttempt(
        user_id=user1.id,
        test_id=refreshed_python_test.id,
        total_questions=len(refreshed_python_test.questions),
    )
    db.session.add(attempt)
    db.session.flush()

    for question in refreshed_python_test.questions:
        correct_answers = [answer for answer in question.answers if answer.is_correct]
        selected_ids = [answer.id for answer in correct_answers]
        if question.question_type == "single":
            selected_ids = selected_ids[:1]
        for answer_id in selected_ids:
            attempt.answers.append(
                UserAnswer(
                    question_id=question.id,
                    selected_answer_id=answer_id,
                    is_correct=True,
                )
            )

    total_questions = len(refreshed_python_test.questions)
    attempt.mark_completed(
        score=refreshed_python_test.grade_for_errors(0),
        correct_answers=total_questions,
        total_questions=total_questions,
    )

    db.session.commit()
